<?php

/*
 * Copyright (c) Fusonic GmbH. All rights reserved.
 * Licensed under the MIT License. See LICENSE file in the project root for license information.
 */

declare(strict_types=1);

namespace Fusonic\OpenGraph\Elements;

use Fusonic\OpenGraph\Property;

/**
 * An OpenGraph video element.
 */
class Video extends ElementBase
{
    /**
     * The URL of a video resource associated with the object.
     */
    public ?string $url = null;

    /**
     * An alternate URL to use if a video resource requires HTTPS.
     */
    public ?string $secureUrl = null;

    /**
     * The MIME type of a video resource associated with the object.
     */
    public ?string $type = null;

    /**
     * The width of a video resource associated with the object in pixels.
     */
    public ?int $width = null;

    /**
     * The height of a video resource associated with the object in pixels.
     */
    public ?int $height = null;

    /**
     * @param string $url URL to the video
     */
    public function __construct(string $url)
    {
        $this->url = $url;
    }

    /**
     * Gets all properties set on this element.
     *
     * @return Property[]
     */
    public function getProperties(): array
    {
        $properties = [];

        // URL must precede all other properties
        if (null !== $this->url) {
            $properties[] = new Property(Property::VIDEO_URL, $this->url);
        }

        if (null !== $this->height) {
            $properties[] = new Property(Property::VIDEO_HEIGHT, $this->height);
        }

        if (null !== $this->secureUrl) {
            $properties[] = new Property(Property::VIDEO_SECURE_URL, $this->secureUrl);
        }

        if (null !== $this->type) {
            $properties[] = new Property(Property::VIDEO_TYPE, $this->type);
        }

        if (null !== $this->width) {
            $properties[] = new Property(Property::VIDEO_WIDTH, $this->width);
        }

        return $properties;
    }
}
