/**
 * SPDX-FileCopyrightText: 2017-2024 Nextcloud GmbH and Nextcloud contributors
 * SPDX-FileCopyrightText: 2015 ownCloud Inc.
 * SPDX-License-Identifier: AGPL-3.0-only
 */

describe('MimeType tests', function() {
	let _files
	let _aliases
	let _theme

	beforeEach(function() {
		_files = OC.MimeTypeList.files
		_aliases = OC.MimeTypeList.aliases
		_theme = OC.MimeTypeList.themes.abc

		OC.MimeTypeList.files = ['folder', 'folder-shared', 'folder-external', 'foo-bar', 'foo', 'file']
		OC.MimeTypeList.aliases = { 'app/foobar': 'foo/bar' }
		OC.MimeTypeList.themes.abc = ['folder']
	})

	afterEach(function() {
		OC.MimeTypeList.files = _files
		OC.MimeTypeList.aliases = _aliases
		OC.MimeTypeList.themes.abc = _theme
	})

	describe('_getFile', function() {
		it('returns the correct icon for "dir"', function() {
			const res = OC.MimeType._getFile('dir', OC.MimeTypeList.files)
			expect(res).toEqual('folder')
		})

		it('returns the correct icon for "dir-shared"', function() {
			const res = OC.MimeType._getFile('dir-shared', OC.MimeTypeList.files)
			expect(res).toEqual('folder-shared')
		})

		it('returns the correct icon for "dir-external"', function() {
			const res = OC.MimeType._getFile('dir-external', OC.MimeTypeList.files)
			expect(res).toEqual('folder-external')
		})

		it('returns the correct icon for a mimetype for which we have an icon', function() {
			const res = OC.MimeType._getFile('foo/bar', OC.MimeTypeList.files)
			expect(res).toEqual('foo-bar')
		})

		it('returns the correct icon for a mimetype for which we only have a general mimetype icon', function() {
			const res = OC.MimeType._getFile('foo/baz', OC.MimeTypeList.files)
			expect(res).toEqual('foo')
		})

		it('return the file mimetype if we have no matching icon but do have a file icon', function() {
			const res = OC.MimeType._getFile('foobar', OC.MimeTypeList.files)
			expect(res).toEqual('file')
		})

		it('return null if we do not have a matching icon', function() {
			const res = OC.MimeType._getFile('xyz', [])
			expect(res).toEqual(null)
		})
	})

	describe('getIconUrl', function() {
		describe('no theme', function() {
			let _themeFolder

			beforeEach(function() {
				_themeFolder = OC.theme.folder
				OC.theme.folder = ''
				// Clear mimetypeIcons caches
				OC.MimeType._mimeTypeIcons = {}
			})

			afterEach(function() {
				OC.theme.folder = _themeFolder
			})

			it('return undefined if the an icon for undefined is requested', function() {
				const res = OC.MimeType.getIconUrl(undefined)
				expect(res).toEqual(undefined)
			})

			it('return the url for the mimetype file', function() {
				const res = OC.MimeType.getIconUrl('file')
				expect(res).toEqual(OC.getRootPath() + '/core/img/filetypes/file.svg')
			})

			it('test if the cache works correctly', function() {
				OC.MimeType._mimeTypeIcons = {}
				expect(Object.keys(OC.MimeType._mimeTypeIcons).length).toEqual(0)

				let res = OC.MimeType.getIconUrl('dir')
				expect(Object.keys(OC.MimeType._mimeTypeIcons).length).toEqual(1)
				expect(OC.MimeType._mimeTypeIcons.dir).toEqual(res)

				res = OC.MimeType.getIconUrl('dir-shared')
				expect(Object.keys(OC.MimeType._mimeTypeIcons).length).toEqual(2)
				expect(OC.MimeType._mimeTypeIcons['dir-shared']).toEqual(res)
			})

			it('test if alaiases are converted correctly', function() {
				const res = OC.MimeType.getIconUrl('app/foobar')
				expect(res).toEqual(OC.getRootPath() + '/core/img/filetypes/foo-bar.svg')
				expect(OC.MimeType._mimeTypeIcons['foo/bar']).toEqual(res)
			})
		})

		describe('themes', function() {
			let _themeFolder

			beforeEach(function() {
				_themeFolder = OC.theme.folder
				OC.theme.folder = 'abc'
				// Clear mimetypeIcons caches
				OC.MimeType._mimeTypeIcons = {}
			})

			afterEach(function() {
				OC.theme.folder = _themeFolder
			})

			it('test if theme path is used if a theme icon is availble', function() {
				const res = OC.MimeType.getIconUrl('dir')
				expect(res).toEqual(OC.getRootPath() + '/themes/abc/core/img/filetypes/folder.svg')
			})

			it('test if we fallback to the default theme if no icon is available in the theme', function() {
				const res = OC.MimeType.getIconUrl('dir-shared')
				expect(res).toEqual(OC.getRootPath() + '/core/img/filetypes/folder-shared.svg')
			})
		})
	})
})
