<?php

/**
 * SPDX-FileCopyrightText: 2016-2024 Nextcloud GmbH and Nextcloud contributors
 * SPDX-FileCopyrightText: 2016 ownCloud, Inc.
 * SPDX-License-Identifier: AGPL-3.0-only
 */
// use OCP namespace for all classes that are considered public.
// This means that they should be used by apps instead of the internal Nextcloud classes

namespace OCP;

use Closure;
use OCP\AppFramework\Attribute\Consumable;
use OCP\Files\File;
use OCP\Files\FileInfo;
use OCP\Files\NotFoundException;
use OCP\Files\SimpleFS\ISimpleFile;
use OCP\Preview\IProviderV2;

/**
 * This class provides functions to render and show thumbnails and previews of files
 * @since 6.0.0
 * @psalm-type ProviderClosure = Closure():(IProviderV2|false)
 */
#[Consumable(since: '6.0.0')]
interface IPreview {
	/**
	 * @since 11.0.0
	 */
	public const MODE_FILL = 'fill';

	/**
	 * @since 11.0.0
	 */
	public const MODE_COVER = 'cover';

	/**
	 * In order to improve lazy loading a closure can be registered which will be
	 * called in case preview providers are actually requested
	 *
	 * @param string $mimeTypeRegex Regex with the mime types that are supported by this provider
	 * @param ProviderClosure $callable
	 * @since 8.1.0
	 * @see \OCP\AppFramework\Bootstrap\IRegistrationContext::registerPreviewProvider
	 *
	 * @deprecated 23.0.0 Register your provider via the IRegistrationContext when booting the app
	 */
	public function registerProvider(string $mimeTypeRegex, Closure $callable): void;

	/**
	 * Get all providers
	 * @return array<string, list<ProviderClosure>>
	 * @since 8.1.0
	 */
	public function getProviders(): array;

	/**
	 * Does the manager have any providers
	 * @since 8.1.0
	 */
	public function hasProviders(): bool;

	/**
	 * Returns a preview of a file
	 *
	 * The cache is searched first and if nothing usable was found then a preview is
	 * generated by one of the providers
	 *
	 * @param IPreview::MODE_* $mode
	 * @param string $mimeType To force a given mimetype for the file (files_versions needs this)
	 * @param bool $cacheResult Whether to cache the preview on the filesystem. Default to true. Can be useful to set to false to limit the amount of stored previews.
	 * @return ISimpleFile
	 * @throws NotFoundException
	 * @throws \InvalidArgumentException if the preview would be invalid (in case the original image is invalid)
	 * @since 11.0.0 - \InvalidArgumentException was added in 12.0.0
	 * @since 32.0.0 - getPreview($cacheResult) added the $cacheResult argument to the signature
	 */
	public function getPreview(File $file, int $width = -1, int $height = -1, bool $crop = false, string $mode = IPreview::MODE_FILL, ?string $mimeType = null, bool $cacheResult = true): ISimpleFile;

	/**
	 * Returns true if the passed mime type is supported
	 * @param string $mimeType A glob
	 * @since 6.0.0
	 */
	public function isMimeSupported(string $mimeType = '*'): bool;

	/**
	 * Check if a preview can be generated for a file
	 *
	 * @param FileInfo $file
	 * @param string|null $mimeType To force a given mimetype for the file
	 * @since 8.0.0
	 * @since 32.0.0 - isAvailable($mimeType) added the $mimeType argument to the signature
	 */
	public function isAvailable(FileInfo $file, ?string $mimeType = null): bool;

	/**
	 * Generates previews of a file
	 *
	 * @param array $specifications
	 * @return ISimpleFile the last preview that was generated
	 * @throws NotFoundException
	 * @throws \InvalidArgumentException if the preview would be invalid (in case the original image is invalid)
	 * @since 19.0.0
	 */
	public function generatePreviews(File $file, array $specifications, ?string $mimeType = null): ISimpleFile;
}
