<?php

/**
 * SPDX-FileCopyrightText: 2017 Nextcloud GmbH and Nextcloud contributors
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */
namespace OC\Repair\Owncloud;

use OCP\DB\QueryBuilder\IQueryBuilder;
use OCP\IConfig;
use OCP\IDBConnection;
use OCP\Migration\IOutput;
use OCP\Migration\IRepairStep;
use OCP\PreConditionNotMetException;

/**
 * Copies the email address from the accounts table to the preference table,
 * before the data structure is changed and the information is gone
 */
class SaveAccountsTableData implements IRepairStep {
	public const BATCH_SIZE = 75;

	protected bool $hasForeignKeyOnPersistentLocks = false;

	public function __construct(
		protected IDBConnection $db,
		protected IConfig $config,
	) {
	}

	public function getName(): string {
		return 'Copy data from accounts table when migrating from ownCloud';
	}

	public function run(IOutput $output): void {
		if (!$this->shouldRun()) {
			return;
		}

		$offset = 0;
		$numUsers = $this->runStep($offset);

		while ($numUsers === self::BATCH_SIZE) {
			$offset += $numUsers;
			$numUsers = $this->runStep($offset);
		}

		// oc_persistent_locks will be removed later on anyways so we can just drop and ignore any foreign key constraints here
		$tableName = $this->config->getSystemValueString('dbtableprefix', 'oc_') . 'persistent_locks';
		$schema = $this->db->createSchema();
		$table = $schema->getTable($tableName);
		foreach ($table->getForeignKeys() as $foreignKey) {
			$table->removeForeignKey($foreignKey->getName());
		}
		$this->db->migrateToSchema($schema);

		// Remove the table
		if ($this->hasForeignKeyOnPersistentLocks) {
			$this->db->dropTable('persistent_locks');
		}
		$this->db->dropTable('accounts');
	}

	protected function shouldRun(): bool {
		$schema = $this->db->createSchema();
		$prefix = $this->config->getSystemValueString('dbtableprefix', 'oc_');

		$tableName = $prefix . 'accounts';
		if (!$schema->hasTable($tableName)) {
			return false;
		}

		$table = $schema->getTable($tableName);
		if (!$table->hasColumn('user_id')) {
			return false;
		}

		if ($schema->hasTable($prefix . 'persistent_locks')) {
			$locksTable = $schema->getTable($prefix . 'persistent_locks');
			$foreignKeys = $locksTable->getForeignKeys();
			foreach ($foreignKeys as $foreignKey) {
				if ($tableName === $foreignKey->getForeignTableName()) {
					$this->hasForeignKeyOnPersistentLocks = true;
				}
			}
		}

		return true;
	}

	/**
	 * @return int Number of copied users
	 */
	protected function runStep(int $offset): int {
		$query = $this->db->getQueryBuilder();
		$query->select('*')
			->from('accounts')
			->orderBy('id')
			->setMaxResults(self::BATCH_SIZE);

		if ($offset > 0) {
			$query->setFirstResult($offset);
		}

		$result = $query->executeQuery();

		$update = $this->db->getQueryBuilder();
		$update->update('users')
			->set('displayname', $update->createParameter('displayname'))
			->where($update->expr()->eq('uid', $update->createParameter('userid')));

		$updatedUsers = 0;
		while ($row = $result->fetch()) {
			try {
				$this->migrateUserInfo($update, $row);
			} catch (PreConditionNotMetException|\UnexpectedValueException) {
				// Ignore and continue
			}
			$updatedUsers++;
		}
		$result->closeCursor();

		return $updatedUsers;
	}

	/**
	 * @throws PreConditionNotMetException
	 * @throws \UnexpectedValueException
	 */
	protected function migrateUserInfo(IQueryBuilder $update, array $userdata): void {
		$state = (int)$userdata['state'];
		if ($state === 3) {
			// Deleted user, ignore
			return;
		}

		if ($userdata['email'] !== null) {
			$this->config->setUserValue($userdata['user_id'], 'settings', 'email', $userdata['email']);
		}
		if ($userdata['quota'] !== null) {
			$this->config->setUserValue($userdata['user_id'], 'files', 'quota', $userdata['quota']);
		}
		if ($userdata['last_login'] !== null) {
			$this->config->setUserValue($userdata['user_id'], 'login', 'lastLogin', $userdata['last_login']);
		}
		if ($state === 1) {
			$this->config->setUserValue($userdata['user_id'], 'core', 'enabled', 'true');
		} elseif ($state === 2) {
			$this->config->setUserValue($userdata['user_id'], 'core', 'enabled', 'false');
		}

		if ($userdata['display_name'] !== null) {
			// user.displayname only allows 64 characters but old accounts.display_name allowed 255 characters
			$update->setParameter('displayname', mb_substr($userdata['display_name'], 0, 64))
				->setParameter('userid', $userdata['user_id']);
			$update->executeStatement();
		}
	}
}
