<?php

/**
 * SPDX-FileCopyrightText: 2016 Nextcloud GmbH and Nextcloud contributors
 * SPDX-License-Identifier: AGPL-3.0-or-later
 */

namespace OCA\FirstRunWizard\Notification;

use OCP\IConfig;
use OCP\IURLGenerator;
use OCP\IUserManager;
use OCP\L10N\IFactory;
use OCP\Notification\IManager as INotificationManager;
use OCP\Notification\INotification;
use OCP\Notification\INotifier;
use OCP\Notification\UnknownNotificationException;
use OCP\User\Backend\ISetPasswordBackend;
use Override;

class Notifier implements INotifier {
	public function __construct(
		private readonly IFactory $factory,
		private readonly IUserManager $userManager,
		private readonly INotificationManager $notificationManager,
		private readonly IURLGenerator $url,
		private readonly IConfig $config,
	) {
	}

	#[Override]
	public function getID(): string {
		return 'firstrunwizard';
	}

	#[Override]
	public function getName(): string {
		return $this->factory->get('firstrunwizard')->t('First run wizard');
	}

	#[Override]
	public function prepare(INotification $notification, string $languageCode): INotification {
		if ($notification->getApp() !== 'firstrunwizard') {
			// Not my app => throw
			throw new UnknownNotificationException();
		}

		switch ($notification->getSubject()) {
			case 'profile':
				$subject = $this->getSubject($notification, $languageCode);
				if ($subject === '') {
					// Everything done, mark the notification as processed...
					$this->notificationManager->markProcessed($notification);
					throw new UnknownNotificationException();
				}

				$notification->setParsedSubject($subject)
					->setLink($this->url->linkToRouteAbsolute('settings.PersonalSettings.index'))
					->setIcon($this->url->getAbsoluteURL($this->url->imagePath('firstrunwizard', 'user.svg')));
				return $notification;
			case 'apphint-calendar':
			case 'apphint-contacts':
			case 'apphint-forms':
			case 'apphint-mail':
			case 'apphint-spreed':
			case 'apphint-tasks':
			case 'apphint-deck':
			case 'apphint-notes':
			case 'apphint-recognize':
			case 'apphint-groupfolders':
			case 'apphint-whiteboard':
				$app = $notification->getObjectId();
				return $this->setAppHintDetails($notification, $languageCode, $app);
			default:
				// Unknown subject => Unknown notification => throw
				throw new UnknownNotificationException();
		}
	}

	protected function getSubject(INotification $notification, string $languageCode): string {
		$l = $this->factory->get('firstrunwizard', $languageCode);
		$user = $this->userManager->get($notification->getUser());
		if ($user === null) {
			return '';
		}

		$email = $user->getEMailAddress();
		if ($email === null || $email === '') {
			if ($this->config->getSystemValue('lost_password_link', '') || !$user->getBackend() instanceof ISetPasswordBackend) {
				return $l->t('Add your profile information! For example your email is needed to receive notifications.');
			}
			return $l->t('Add your profile information! For example your email is needed to receive notifications and reset your password.');
		}

		if ($user->canChangeDisplayName() && $user->getDisplayName() === $user->getUID()) {
			if ($user->canChangeAvatar() && $user->getAvatarImage(32) === null) {
				return $l->t('Add your profile information! Set a profile picture and full name for easier recognition across all features.');
			} else {
				return $l->t('Add your profile information! Set a full name for easier recognition across all features.');
			}
		} else {
			if ($user->canChangeAvatar() && $user->getAvatarImage(32) === null) {
				return $l->t('Add your profile information! Set a profile picture for easier recognition across all features.');
			} else {
				return '';
			}
		}
	}

	protected function setAppHintDetails(INotification $notification, string $languageCode, string $app): INotification {
		$l = $this->factory->get('firstrunwizard', $languageCode);
		$appLink = '';
		switch ($app) {
			case 'calendar':
				$notification->setParsedSubject($l->t('App recommendation: Calendar'));
				$notification->setParsedMessage($l->t('Schedule work & meetings, synced with all your devices.'));
				$appLink = '/organization/calendar';
				break;
			case 'contacts':
				$notification->setParsedSubject($l->t('App recommendation: Contacts'));
				$notification->setParsedMessage($l->t('Keep your colleagues and friends in one place without leaking their private info.'));
				$appLink = '/organization/contacts';
				break;
			case 'mail':
				$notification->setParsedSubject($l->t('App recommendation: Mail'));
				$notification->setParsedMessage($l->t('Simple email app nicely integrated with Files, Contacts and Calendar.'));
				$appLink = '/social/mail';
				break;

			case 'tasks':
				$notification->setParsedSubject($l->t('App recommendation: Tasks'));
				$notification->setParsedMessage($l->t('Sync tasks from various devices with your Nextcloud and edit them online.'));
				$appLink = '/organization/tasks';
				break;
			case 'deck':
				$notification->setParsedSubject($l->t('App recommendation: Deck'));
				$notification->setParsedMessage($l->t('Kanban style organization for personal planning and team projects.'));
				$appLink = '/organization/deck';
				break;
			case 'forms':
				$notification->setParsedSubject($l->t('App recommendation: Forms'));
				$notification->setParsedMessage($l->t('Simple surveys and questionnaires, self-hosted'));
				$appLink = '/organization/forms';
				break;
			case 'recognize':
				$notification->setParsedSubject($l->t('App recommendation: Recognize'));
				$notification->setParsedMessage($l->t('Smart media tagging for Nextcloud'));
				$appLink = '/organization/recognize';
				break;
			case 'groupfolders':
				$notification->setParsedSubject($l->t('App recommendation: Group folders'));
				$notification->setParsedMessage($l->t('Admin-configured folders shared by everyone in a group.'));
				$appLink = '/files/groupfolders';
				break;
			case 'whiteboard':
				$notification->setParsedSubject($l->t('App recommendation: Whiteboard'));
				$notification->setParsedMessage($l->t('Create and share whiteboards with others and collaborate in real-time.'));
				$appLink = '/office/whiteboard';
				break;
		}
		$notification
			->setLink($this->url->linkToRouteAbsolute('settings.AppSettings.viewApps') . $appLink)
			->setIcon($this->url->getAbsoluteURL($this->url->imagePath('firstrunwizard', 'apps/' . $app . '.svg')));
		return $notification;
	}
}
